'use client';

import { motion } from 'framer-motion';
import Image from 'next/image';
import Link from 'next/link';
import Navbar from '../../components/Navbar';
import SubscribeSection from '../../components/SubscribeSection';
import RecentBlogSection from '@/app/components/RecentBlogSection';
import { use, useCallback, useEffect, useState } from 'react';
import { client } from "@/sanity/client";
import { PortableText, type SanityDocument, defineQuery } from "next-sanity";
import imageUrlBuilder from "@sanity/image-url";
import type { SanityImageSource } from "@sanity/image-url/lib/types/types";
import { BlogPostProps } from '@/app/components/BlogPost';


const BLOG_POSTS_QUERY = defineQuery(`*[_type == "blogPost"] {
    title,
    excerpt,
    "slug": slug.current,
    "image": mainImage.asset->url,
    "author": author->name,
    "authorImage": author->image.asset->url,
    _createdAt,
  }[0...3] | order(_createdAt desc)`);


const POST_QUERY = `*[_type == "blogPost" && slug.current == $slug][0]{
  "id": _id,
  title,
  content,
  "slug": slug.current,
  "image": mainImage.asset,
  "author": author->name,
  "authorImage": author->image.asset->url,
  "date": _createdAt,
  excerpt,
  viewsCount
}`;

const { projectId, dataset } = client.config();
const urlFor = (source: SanityImageSource) =>
    projectId && dataset
        ? imageUrlBuilder({ projectId, dataset }).image(source)
        : null;

export default function BlogPost({ params }: { params: Promise<{ slug: string }> }) {
    const [post, setPost] = useState<SanityDocument | null>(null);
    const [blogPosts, setBlogPosts] = useState<BlogPostProps[]>([]);
    const [isLoading, setIsLoading] = useState(true);
    const [isLoadingBlogPosts, setIsLoadingBlogPosts] = useState(true);
    const [error, setError] = useState<string | null>(null);

    const { slug } = use(params);

    // Initial load
    useEffect(() => {
        async function fetchPosts() {
            try {
                const data = await client.fetch(BLOG_POSTS_QUERY, {}, {});
                setBlogPosts(data || []);
            } catch (err) {
                console.error('Error fetching blog posts:', err);
                // setError('Failed to load blog posts');
            } finally {
                setIsLoadingBlogPosts(false);
            }
        }

        fetchPosts();
    }, []);

    useEffect(() => {
        const fetchPost = async () => {
            try {
                setIsLoading(true);
                const fetchedPost = await client.fetch<SanityDocument>(POST_QUERY, { slug: slug }, {});

                if (!fetchedPost) {
                    setError('Post not found');
                    return;
                }

                setPost(fetchedPost);
            } catch (err) {
                setError('Failed to load blog post');
                console.error('Error fetching post:', err);
            } finally {
                setIsLoading(false);
            }
        };

        fetchPost();
    }, [slug]);

    useEffect(() => {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        const formattedBlogPosts: BlogPostProps[] = blogPosts.map((_post: any) => ({
            ..._post,
            date: new Date(_post._createdAt).toLocaleDateString('en-US', {
                day: 'numeric',
                month: 'short',
                year: 'numeric'
            })
        }));

        setBlogPosts(formattedBlogPosts);
    }, [isLoadingBlogPosts]);

    const incrementPostViews = useCallback(
        async (slug: string) => {
            const storageKey = `blog_${slug}_viewed`;
            const lastViewed = localStorage.getItem(storageKey);

            if (lastViewed) {
                const lastViewedDate = new Date(lastViewed);
                const thirtyDaysAgo = new Date();
                thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);

                if (lastViewedDate > thirtyDaysAgo) {
                    return; // Don't increment if viewed within last 30 days
                }
            }

            try {
                await fetch('/api/update-views', {
                    method: 'POST',
                    body: JSON.stringify({ docId: post?.id })
                })
                localStorage.setItem(storageKey, new Date().toISOString());
            } catch (err) {
                console.error('Failed to increment view count:', err);
            }
        },
        [post]
    );

    // Add this effect to increment views when the post loads
    useEffect(() => {
        if (post?.slug) {
            incrementPostViews(post.slug);
        }
    }, [post?.slug, incrementPostViews]);

    if (isLoading) {
        return (
            <main>
                <Navbar topPlacement={true} />
                <div className="mt-28 container mx-auto px-4 md:max-w-[1200px] min-h-screen flex items-center justify-center">
                    <div className="animate-pulse text-xl">Loading post...</div>
                </div>
            </main>
        );
    }

    if (error || !post) {
        return (
            <main>
                <Navbar topPlacement={true} />
                <div className="mt-28 container mx-auto px-4 md:max-w-[1200px] min-h-screen flex flex-col items-center justify-center gap-4">
                    <h1 className="text-2xl font-bold text-red-600">{error || 'Post not found'}</h1>
                    <Link href="/blog" className="text-blue-600 hover:underline">
                        ← Back to blog
                    </Link>
                </div>
            </main>
        );
    }

    const postImageUrl = post.image
        ? urlFor(post.image)?.width(1920).height(1080).url()
        : '/gallery/2.jpg'; // fallback image

    const postDate = post.date ? new Date(post.date).toLocaleDateString('en-US', {
        day: 'numeric',
        month: 'short',
        year: 'numeric'
    }) : null


    return (
        <main>
            <Navbar topPlacement={true} />

            <div className="mt-28 container mx-auto px-4 md:max-w-[1200px]  ">
                {/* Hero Image */}
                <div className="relative w-full h-[400px] mb-8">
                    <Image
                        src={postImageUrl!}
                        alt={post.title || 'Blog post hero image'}
                        fill
                        className="object-cover brightness-90 rounded-2xl"
                    />
                </div>

                {/* Content Container */}
                <div className="px-2 md:px-20 ">
                    {/* Blog Header */}
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        className=" mb-12"
                    >
                        <h1 className="text-3xl font-script md:text-4xl mb-2">
                            {post.title}
                        </h1>
                        <div className="flex items-center justify-start gap-2 text-sm text-[#222222]/80">
                            <div className="flex items-center gap-2">
                                <Image
                                    src={post.authorImage}
                                    alt={post.author}
                                    width={32}
                                    height={32}
                                    className="rounded-full aspect-square object-cover"
                                />
                                <span>{post.author}</span>
                            </div>
                            <span>•</span>
                            {postDate && <span>{postDate}</span>}

                            {post.viewsCount && <><span>•</span><span>{post.viewsCount} views</span></>}
                        </div>
                    </motion.div>

                    {/* Blog Content */}
                    <motion.div
                        initial={{ opacity: 0 }}
                        animate={{ opacity: 1 }}
                        transition={{ delay: 0.2 }}
                        className="prose w-full mb-16"
                    // dangerouslySetInnerHTML={{ __html: post.content }}
                    >
                        {Array.isArray(post.content) && <PortableText
                            value={post.content}
                            components={{
                                block: {
                                    h1: ({ children }) => <h1 className="text-3xl font-bold my-4">{children}</h1>,
                                    h2: ({ children }) => <h2 className="text-2xl font-bold my-3">{children}</h2>,
                                    h3: ({ children }) => <h3 className="text-xl font-bold my-2">{children}</h3>,
                                    normal: ({ children }) => <p className="my-2">{children}</p>,
                                },
                                list: {
                                    bullet: ({ children }) => <ul className="list-disc pl-4 my-2">{children}</ul>,
                                    number: ({ children }) => <ol className="list-decimal pl-4 my-2">{children}</ol>,
                                },
                                marks: {
                                    strong: ({ children }) => <strong className="font-bold">{children}</strong>,
                                    em: ({ children }) => <em className="italic">{children}</em>,
                                    link: ({ value, children }) => (
                                        <a href={value?.href} className="text-coral hover:underline">
                                            {children}
                                        </a>
                                    ),
                                },
                                types: {
                                    image: ({ value, isInline }) =>
                                        // eslint-disable-next-line @next/next/no-img-element
                                        <img src={(urlFor(value)?.width(isInline ? 100 : 1080).height(isInline ? 400 : 500).fit('max').auto('format').url()) || ''} alt={value?.alt || ''} />
                                },
                            }}
                        />}
                    </motion.div>

                    {/* Share Section */}
                    {/* <div className="flex items-center justify-end gap-4 mb-16">
                        <span className="text-sm">Share this:</span>
                        <div className="flex gap-2">
                            <Link href="#" className="text-gray-600 hover:text-gray-900">
                                <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 24 24" aria-hidden="true">
                                    <path d="M22 12c0-5.523-4.477-10-10-10S2 6.477 2 12c0 4.991 3.657 9.128 8.438 9.878v-6.987h-2.54V12h2.54V9.797c0-2.506 1.492-3.89 3.777-3.89 1.094 0 2.238.195 2.238.195v2.46h-1.26c-1.243 0-1.63.771-1.63 1.562V12h2.773l-.443 2.89h-2.33v6.988C18.343 21.128 22 16.991 22 12z" />
                                </svg>
                            </Link>
                            <Link href="#" className="text-gray-600 hover:text-gray-900">
                                <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 24 24" aria-hidden="true">
                                    <path d="M8.29 20.251c7.547 0 11.675-6.253 11.675-11.675 0-.178 0-.355-.012-.53A8.348 8.348 0 0022 5.92a8.19 8.19 0 01-2.357.646 4.118 4.118 0 001.804-2.27 8.224 8.224 0 01-2.605.996 4.107 4.107 0 00-6.993 3.743 11.65 11.65 0 01-8.457-4.287 4.106 4.106 0 001.27 5.477A4.072 4.072 0 012.8 9.713v.052a4.105 4.105 0 003.292 4.022 4.095 4.095 0 01-1.853.07 4.108 4.108 0 003.834 2.85A8.233 8.233 0 012 18.407a11.616 11.616 0 006.29 1.84" />
                                </svg>
                            </Link>
                            <Link href="#" className="text-gray-600 hover:text-gray-900">
                                <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 24 24" aria-hidden="true">
                                    <path d="M12.315 2c2.43 0 2.784.013 3.808.06 1.064.049 1.791.218 2.427.465a4.902 4.902 0 011.772 1.153 4.902 4.902 0 011.153 1.772c.247.636.416 1.363.465 2.427.048 1.067.06 1.407.06 4.123v.08c0 2.643-.012 2.987-.06 4.043-.049 1.064-.218 1.791-.465 2.427a4.902 4.902 0 01-1.153 1.772 4.902 4.902 0 01-1.772 1.153c-.636.247-1.363.416-2.427.465-1.067.048-1.407.06-4.123.06h-.08c-2.643 0-2.987-.012-4.043-.06-1.064-.049-1.791-.218-2.427-.465a4.902 4.902 0 01-1.772-1.153 4.902 4.902 0 01-1.153-1.772c-.247-.636-.416-1.363-.465-2.427-.047-1.024-.06-1.379-.06-3.808v-.63c0-2.43.013-2.784.06-3.808.049-1.064.218-1.791.465-2.427a4.902 4.902 0 011.153-1.772A4.902 4.902 0 015.45 2.525c.636-.247 1.363-.416 2.427-.465C8.901 2.013 9.256 2 11.685 2h.63zm-.081 1.802h-.468c-2.456 0-2.784.011-3.807.058-.975.045-1.504.207-1.857.344-.467.182-.8.398-1.15.748-.35.35-.566.683-.748 1.15-.137.353-.3.882-.344 1.857-.047 1.023-.058 1.351-.058 3.807v.468c0 2.456.011 2.784.058 3.807.045.975.207 1.504.344 1.857.182.466.399.8.748 1.15.35.35.683.566 1.15.748.353.137.882.3 1.857.344 1.054.048 1.37.058 4.041.058h.08c2.597 0 2.917-.01 3.96-.058.976-.045 1.505-.207 1.858-.344.466-.182.8-.398 1.15-.748.35-.35.566-.683.748-1.15.137-.353.3-.882.344-1.857.048-1.055.058-1.37.058-4.041v-.08c0-2.597-.01-2.917-.058-3.96-.045-.976-.207-1.505-.344-1.858a3.097 3.097 0 00-.748-1.15 3.098 3.098 0 00-1.15-.748c-.353-.137-.882-.3-1.857-.344-1.023-.047-1.351-.058-3.807-.058zM12 6.865a5.135 5.135 0 110 10.27 5.135 5.135 0 010-10.27zm0 1.802a3.333 3.333 0 100 6.666 3.333 3.333 0 000-6.666zm5.338-3.205a1.2 1.2 0 110 2.4 1.2 1.2 0 010-2.4z" />
                                </svg>
                            </Link>
                        </div>
                    </div> */}
                </div>

                {/* Other posts */}
                <RecentBlogSection title="Other Blog Posts" loadMore={false} limit={3} blogPosts={blogPosts} />


            </div>

            {/* Subscribe Section */}
            <SubscribeSection />
        </main>
    );
}